/*____________________________________________________________________________
	Copyright (C) 1997 Network Associates Inc. and affiliated companies.
	All rights reserved.

	$Id: pflPrefTypes.c,v 1.5 1999/10/13 15:22:24 dgal Exp $
____________________________________________________________________________*/
#include <string.h>

#include "pflPrefTypes.h"
#include "pgpPFLErrors.h"
#include "pgpDebug.h"
#include "pgpMem.h"
#include "pgpMemoryMgr.h"
#include "pgpEndianConversion.h"

#include "pgpPFLPriv.h"

PGPError PGPGetPrefBoolean(PGPPrefRef prefRef,
							   PGPPrefIndex prefIndex,
							   PGPBoolean *data)
{
	void *			tempData;
	PGPSize			dataLength;
	PGPError		err;

	if (IsntNull(data))
		*data = FALSE;
		
	PGPValidatePtr(data);

	err = PGPGetPrefData(prefRef, prefIndex, &dataLength, &tempData);
	if (IsPGPError(err))
		return err;

	pgpAssert(dataLength == sizeof(PGPBoolean));
	if (dataLength != sizeof(PGPBoolean))
	{
		PGPFreeData(tempData);
		return kPGPError_BadParams;
	}

	*data = *(PGPBoolean *)tempData;
	PGPFreeData(tempData);

	return err;
}


PGPError PGPSetPrefBoolean(PGPPrefRef prefRef,
							   PGPPrefIndex prefIndex,
							   PGPBoolean data)
{
	PGPError err;

	err = PGPSetPrefData(prefRef, prefIndex, sizeof(PGPBoolean), &data);

	return err;
}


PGPError PGPGetPrefNumber(PGPPrefRef prefRef,
							  PGPPrefIndex prefIndex,
							  PGPUInt32 *data)
{
	void *		tempData;
	PGPSize		dataLength;
	PGPError	err;

	if (IsntNull(data))
		*data = 0;

	PGPValidatePtr(data);

	err = PGPGetPrefData(prefRef, prefIndex, &dataLength, &tempData);
	if (IsPGPError(err))
		return err;

	pgpAssert(dataLength == sizeof(PGPUInt32));
	if (dataLength != sizeof(PGPUInt32))
	{
		PGPFreeData(tempData);
		return kPGPError_BadParams;
	}

	*data = *(PGPUInt32 *)tempData;
	PGPFreeData(tempData);

	return err;
}


PGPError PGPSetPrefNumber(PGPPrefRef prefRef,
							  PGPPrefIndex prefIndex,
							  PGPUInt32 data)
{
	PGPError	err;

	err = PGPSetPrefData(prefRef, prefIndex, sizeof(PGPUInt32),
			(void *) &data);

	return err;
}


PGPError PGPGetPrefStringAlloc(PGPPrefRef prefRef,
								   PGPPrefIndex prefIndex,
								  char **string)
{
	PGPError	err;
	void *		data			= NULL;
	PGPSize		stringLength	= 0;

	if (IsntNull(string))
		*string = NULL;

	PGPValidatePtr(string);

	err = PGPGetPrefData(prefRef, prefIndex, &stringLength, &data);
	if (IsntPGPError(err))
	{
		*string	= (char *)data;
	}
	
	return err;
}


PGPError PGPGetPrefStringBuffer(PGPPrefRef prefRef,
									PGPPrefIndex prefIndex,
									PGPSize maxSize,
									char *string)
{
	PGPError	err;
	void *		tempString;
	PGPSize		tempLength;

	if (IsntNull(string) && (maxSize > 0))
		string[0] = '\0';

	PGPValidatePtr(string);
	PGPValidateParam(maxSize > 0);

	err = PGPGetPrefData(prefRef, prefIndex, &tempLength, &tempString);
	if (IsPGPError(err))
		return err;

	if (tempLength > maxSize)
	{
		tempLength = maxSize;
		string[tempLength-1] = 0;
	}

	pgpCopyMemory(tempString, string, tempLength);
	PGPFreeData(tempString);
	
	return err;
}


PGPError PGPSetPrefString(PGPPrefRef prefRef,
							  PGPPrefIndex prefIndex,
							  const char *string)
{
	PGPError	err;
	PGPSize		stringLength;

	PGPValidatePtr(string);
	
	stringLength = strlen(string)+1;	/* Include null terminator */

	err = PGPSetPrefData(prefRef, prefIndex, stringLength, string);
	return err;
}


PGPError PGPGetPrefStruct(PGPPrefRef prefRef,
							PGPPrefIndex prefIndex,
							PGPPrefStruct **structPtr)
{
	PGPError	err;
	void *		tempBuffer;
	PGPSize		tempLength;

	if (IsntNull(structPtr))
		*structPtr = NULL;

	PGPValidatePtr(structPtr);

	err = PGPGetPrefData(prefRef, prefIndex, &tempLength, &tempBuffer);
	
	if (IsntPGPError(err))
	{
		pgpAssert(tempLength == sizeof(PGPPrefStruct));
		if (tempLength != sizeof(PGPPrefStruct))
		{
			PGPFreeData(tempBuffer);
			err = kPGPError_BadParams;
		}
	}

	if (IsntPGPError(err))
	{
		*structPtr = (PGPPrefStruct *) tempBuffer;
		(*structPtr)->dirty = FALSE;
	}

	return err;
}


PGPError PGPSetPrefStruct(PGPPrefRef prefRef,
							PGPPrefIndex prefIndex,
							PGPPrefStruct *structPtr)
{
	PGPError err;

	PGPValidatePtr(structPtr);

	err = PGPSetPrefData(prefRef, prefIndex, sizeof(PGPPrefStruct), 
			structPtr);
	
	if (IsntPGPError(err))
		structPtr->dirty = FALSE;

	return err;
}


PGPError PGPGetPrefArray(PGPPrefRef prefRef,
							PGPPrefIndex prefIndex,
							PGPPrefArray **arrayPtr)
{
	PGPError				err;
	void *					tempBuffer;
	PGPSize					tempLength;

	if (IsntNull(arrayPtr))
		*arrayPtr = NULL;

	PGPValidatePtr(arrayPtr);

	err = PGPGetPrefData(prefRef, prefIndex, &tempLength, &tempBuffer);
	
	if (IsntPGPError(err))
	{
		pgpAssert(tempLength == sizeof(PGPPrefArray));
		if (tempLength != sizeof(PGPPrefArray))
		{
			PGPFreeData(tempBuffer);
			err = kPGPError_BadParams;
		}
	}

	if (IsntPGPError(err))
	{
		*arrayPtr = (PGPPrefArray *) tempBuffer;
		(*arrayPtr)->dirty = FALSE;
	}

	return err;
}


PGPError PGPSetPrefArray(PGPPrefRef prefRef,
							PGPPrefIndex prefIndex,
							PGPPrefArray *arrayPtr)
{
	PGPError err;

	PGPValidatePtr(arrayPtr);

	err = PGPSetPrefData(prefRef, prefIndex, sizeof(PGPPrefArray), 
			arrayPtr);
	
	if (IsntPGPError(err))
		arrayPtr->dirty = FALSE;

	return err;
}


PGPError PGPGetPrefArrayBoolean(PGPPrefRef prefRef,
									PGPPrefArray *prefArray,
									PGPUInt32 arrayIndex,
									PGPBoolean *data)
{
	PGPValidatePref(prefRef);
	PGPValidatePtr(prefArray);
	PGPValidatePtr(data);

	if (prefArray->type != kPGPPrefType_Boolean)
		return kPGPError_BadParams;

	if (arrayIndex >= prefArray->numElements)
		return kPGPError_BadParams;

	*data = (PGPBoolean) prefArray->elements[arrayIndex].data;
	return kPGPError_NoErr;
}


PGPError PGPSetPrefArrayBoolean(PGPPrefRef prefRef,
									PGPPrefArray *prefArray,
									PGPUInt32 arrayIndex,
									PGPBoolean data)
{
	PGPMemoryMgrRef memoryMgr;
	PGPError err = kPGPError_NoErr;

	PGPValidatePref(prefRef);
	PGPValidatePtr(prefArray);

	if (prefArray->type != kPGPPrefType_Boolean)
		return kPGPError_BadParams;

	PGPPeekPrefMemoryMgr(prefRef, &memoryMgr);

	if (arrayIndex >= prefArray->numElements)
	{
		err = PGPReallocData(memoryMgr,
				&(prefArray->elements),
				sizeof(PGPPrefArrayElement) * (arrayIndex+1),
				0);

		if (IsPGPError(err))
			return err;
		prefArray->numElements = arrayIndex + 1;
	}

	prefArray->elements[arrayIndex].data = (void *) data;
	prefArray->elements[arrayIndex].size = sizeof(PGPBoolean);
	prefArray->dirty = TRUE;
	return kPGPError_NoErr;
}


PGPError PGPGetPrefArrayNumber(PGPPrefRef prefRef,
									PGPPrefArray *prefArray,
									PGPUInt32 arrayIndex,
									PGPUInt32 *data)
{
	PGPValidatePref(prefRef);
	PGPValidatePtr(prefArray);
	PGPValidatePtr(data);

	if (prefArray->type != kPGPPrefType_Number)
		return kPGPError_BadParams;

	if (arrayIndex >= prefArray->numElements)
		return kPGPError_BadParams;

	*data = (PGPUInt32) prefArray->elements[arrayIndex].data;
	return kPGPError_NoErr;
}


PGPError PGPSetPrefArrayNumber(PGPPrefRef prefRef,
									PGPPrefArray *prefArray,
									PGPUInt32 arrayIndex,
									PGPUInt32 data)
{
	PGPMemoryMgrRef memoryMgr;
	PGPError err = kPGPError_NoErr;

	PGPValidatePref(prefRef);
	PGPValidatePtr(prefArray);

	if (prefArray->type != kPGPPrefType_Number)
		return kPGPError_BadParams;

	PGPPeekPrefMemoryMgr(prefRef, &memoryMgr);

	if (arrayIndex >= prefArray->numElements)
	{
		err = PGPReallocData(memoryMgr,
				&(prefArray->elements),
				sizeof(PGPPrefArrayElement) * (arrayIndex+1),
				0);

		if (IsPGPError(err))
			return err;
		prefArray->numElements = arrayIndex + 1;
	}

	prefArray->elements[arrayIndex].data = (void *) data;
	prefArray->elements[arrayIndex].size = sizeof(PGPUInt32);
	prefArray->dirty = TRUE;
	return kPGPError_NoErr;
}


PGPError PGPGetPrefArrayString(PGPPrefRef prefRef,
									PGPPrefArray *prefArray,
									PGPUInt32 arrayIndex,
									char **string)
{
	PGPMemoryMgrRef memoryMgr;

	if (IsntNull(string))
		*string = NULL;

	PGPValidatePref(prefRef);
	PGPValidatePtr(prefArray);
	PGPValidatePtr(string);

	if (prefArray->type != kPGPPrefType_String)
		return kPGPError_BadParams;

	if (arrayIndex >= prefArray->numElements)
		return kPGPError_BadParams;

	PGPPeekPrefMemoryMgr(prefRef, &memoryMgr);

	if (IsntNull(prefArray->elements[arrayIndex].data))
	{
		*string = (char *) PGPNewData(memoryMgr,
							strlen(prefArray->elements[arrayIndex].data)+1,
							kPGPMemoryMgrFlags_Clear);

		if (IsNull(*string))
			return kPGPError_OutOfMemory;

		pgpCopyMemory(prefArray->elements[arrayIndex].data, *string,
			strlen(prefArray->elements[arrayIndex].data) + 1);
	}

	return kPGPError_NoErr;
}


PGPError PGPSetPrefArrayString(PGPPrefRef prefRef,
									PGPPrefArray *prefArray,
									PGPUInt32 arrayIndex,
									char *string)
{
	PGPMemoryMgrRef memoryMgr;
	PGPError err = kPGPError_NoErr;

	PGPValidatePref(prefRef);
	PGPValidatePtr(prefArray);
	PGPValidatePtr(string);

	if (prefArray->type != kPGPPrefType_String)
		return kPGPError_BadParams;

	PGPPeekPrefMemoryMgr(prefRef, &memoryMgr);

	if (arrayIndex >= prefArray->numElements)
	{
		err = PGPReallocData(memoryMgr,
				&(prefArray->elements),
				sizeof(PGPPrefArrayElement) * (arrayIndex+1),
				0);

		if (IsPGPError(err))
			return err;
	}

	if (IsntNull(prefArray->elements[arrayIndex].data))
		PGPFreeData(prefArray->elements[arrayIndex].data);

	prefArray->elements[arrayIndex].data = PGPNewData(memoryMgr,
												strlen(string) + 1,
												kPGPMemoryMgrFlags_Clear);

	if (IsNull(prefArray->elements[arrayIndex].data))
		return kPGPError_OutOfMemory;

	pgpCopyMemory(string, prefArray->elements[arrayIndex].data,
		strlen(string) + 1);

	prefArray->dirty = TRUE;
	return kPGPError_NoErr;
}


PGPError PGPGetPrefArrayByte(PGPPrefRef prefRef,
									PGPPrefArray *prefArray,
									PGPUInt32 arrayIndex,
									PGPSize *dataLength,
									void **data)
{
	PGPMemoryMgrRef memoryMgr;

	if (IsntNull(dataLength))
		*dataLength = 0;
	if (IsntNull(data))
		*data = NULL;

	PGPValidatePref(prefRef);
	PGPValidatePtr(prefArray);
	PGPValidatePtr(dataLength);
	PGPValidatePtr(data);

	if (prefArray->type != kPGPPrefType_Byte)
		return kPGPError_BadParams;

	if (arrayIndex >= prefArray->numElements)
		return kPGPError_BadParams;

	PGPPeekPrefMemoryMgr(prefRef, &memoryMgr);

	if (IsntNull(prefArray->elements[arrayIndex].data))
	{
		*data = PGPNewData(memoryMgr,
					prefArray->elements[arrayIndex].size,
					kPGPMemoryMgrFlags_Clear);

		if (IsNull(*data))
			return kPGPError_OutOfMemory;

		*dataLength = prefArray->elements[arrayIndex].size;

		pgpCopyMemory(prefArray->elements[arrayIndex].data, *data,
			prefArray->elements[arrayIndex].size);
	}

	return kPGPError_NoErr;
}


PGPError PGPSetPrefArrayByte(PGPPrefRef prefRef,
									PGPPrefArray *prefArray,
									PGPUInt32 arrayIndex,
									PGPSize dataLength,
									void *data)
{
	PGPMemoryMgrRef memoryMgr;
	PGPError err = kPGPError_NoErr;

	PGPValidatePref(prefRef);
	PGPValidatePtr(prefArray);

	if (prefArray->type != kPGPPrefType_Byte)
		return kPGPError_BadParams;

	PGPPeekPrefMemoryMgr(prefRef, &memoryMgr);

	if (arrayIndex >= prefArray->numElements)
	{
		err = PGPReallocData(memoryMgr,
				&(prefArray->elements),
				sizeof(PGPPrefArrayElement) * (arrayIndex+1),
				0);

		if (IsPGPError(err))
			return err;
		prefArray->numElements = arrayIndex + 1;
	}

	if (IsntNull(prefArray->elements[arrayIndex].data))
	{
		PGPFreeData(prefArray->elements[arrayIndex].data);
		prefArray->elements[arrayIndex].data = NULL;
	}

	if (IsNull(data))
		return kPGPError_NoErr;

	prefArray->elements[arrayIndex].data = PGPNewData(memoryMgr,
												dataLength,
												kPGPMemoryMgrFlags_Clear);

	if (IsNull(prefArray->elements[arrayIndex].data))
		return kPGPError_OutOfMemory;

	pgpCopyMemory(data, prefArray->elements[arrayIndex].data,
		dataLength);

	prefArray->elements[arrayIndex].size = dataLength;
	prefArray->dirty = TRUE;

	return kPGPError_NoErr;
}


PGPError PGPGetPrefArrayStruct(PGPPrefRef prefRef,
									PGPPrefArray *prefArray,
									PGPUInt32 arrayIndex,
									PGPPrefStruct **prefStruct)
{
	if (IsntNull(prefStruct))
		*prefStruct = NULL;

	PGPValidatePref(prefRef);
	PGPValidatePtr(prefArray);
	PGPValidatePtr(prefStruct);

	if (prefArray->type != kPGPPrefType_Struct)
		return kPGPError_BadParams;

	if (arrayIndex >= prefArray->numElements)
		return kPGPError_BadParams;

	*prefStruct = (PGPPrefStruct *) prefArray->elements[arrayIndex].data;
	return kPGPError_NoErr;
}


PGPError PGPSetPrefArrayStruct(PGPPrefRef prefRef,
									PGPPrefArray *prefArray,
									PGPUInt32 arrayIndex,
									PGPPrefStruct *prefStruct)
{
	PGPMemoryMgrRef memoryMgr;
	PGPError err = kPGPError_NoErr;

	PGPValidatePref(prefRef);
	PGPValidatePtr(prefArray);
	PGPValidatePtr(prefStruct);

	if (prefArray->type != kPGPPrefType_Struct)
		return kPGPError_BadParams;

	PGPPeekPrefMemoryMgr(prefRef, &memoryMgr);

	if (arrayIndex >= prefArray->numElements)
	{
		err = PGPReallocData(memoryMgr,
				&(prefArray->elements),
				sizeof(PGPPrefArrayElement) * (arrayIndex+1),
				0);

		if (IsPGPError(err))
			return err;
		prefArray->numElements = arrayIndex + 1;
	}

	if (prefStruct != (PGPPrefStruct *) prefArray->elements[arrayIndex].data)
	{
		if (IsntNull(prefArray->elements[arrayIndex].data))
			PGPFreePrefStruct(prefArray->elements[arrayIndex].data);

		prefArray->elements[arrayIndex].data = prefStruct;
		prefStruct->dirty = FALSE;
	}

	prefArray->dirty = TRUE;
	return kPGPError_NoErr;
}


PGPError PGPGetPrefStructBoolean(PGPPrefRef prefRef,
									PGPPrefStruct *prefStruct,
									PGPUInt16 memberIndex,
									PGPBoolean *data)
{
	PGPValidatePref(prefRef);
	PGPValidatePtr(prefStruct);
	PGPValidatePtr(data);

	if (memberIndex >= prefStruct->numMembers)
		return kPGPError_BadParams;

	if (prefStruct->members[memberIndex].type != kPGPPrefType_Boolean)
		return kPGPError_BadParams;

	*data = (PGPBoolean) prefStruct->members[memberIndex].data;
	return kPGPError_NoErr;
}


PGPError PGPSetPrefStructBoolean(PGPPrefRef prefRef,
									PGPPrefStruct *prefStruct,
									PGPUInt16 memberIndex,
									PGPBoolean data)
{
	PGPError err = kPGPError_NoErr;

	PGPValidatePref(prefRef);
	PGPValidatePtr(prefStruct);

	if (memberIndex >= prefStruct->numMembers)
		return kPGPError_BadParams;

	if (prefStruct->members[memberIndex].type != kPGPPrefType_Boolean)
		return kPGPError_BadParams;

	prefStruct->members[memberIndex].data = (void *) data;
	prefStruct->dirty = TRUE;
	return kPGPError_NoErr;
}


PGPError PGPGetPrefStructNumber(PGPPrefRef prefRef,
									PGPPrefStruct *prefStruct,
									PGPUInt16 memberIndex,
									PGPUInt32 *data)
{
	PGPValidatePref(prefRef);
	PGPValidatePtr(prefStruct);
	PGPValidatePtr(data);

	if (memberIndex >= prefStruct->numMembers)
		return kPGPError_BadParams;

	if (prefStruct->members[memberIndex].type != kPGPPrefType_Number)
		return kPGPError_BadParams;

	*data = (PGPUInt32) prefStruct->members[memberIndex].data;
	return kPGPError_NoErr;
}


PGPError PGPSetPrefStructNumber(PGPPrefRef prefRef,
									PGPPrefStruct *prefStruct,
									PGPUInt16 memberIndex,
									PGPUInt32 data)
{
	PGPError err = kPGPError_NoErr;

	PGPValidatePref(prefRef);
	PGPValidatePtr(prefStruct);

	if (memberIndex >= prefStruct->numMembers)
		return kPGPError_BadParams;

	if (prefStruct->members[memberIndex].type != kPGPPrefType_Number)
		return kPGPError_BadParams;

	prefStruct->members[memberIndex].data = (void *) data;
	prefStruct->dirty = TRUE;
	return kPGPError_NoErr;
}


PGPError PGPGetPrefStructString(PGPPrefRef prefRef,
									PGPPrefStruct *prefStruct,
									PGPUInt16 memberIndex,
									char **string)
{
	PGPMemoryMgrRef memoryMgr;

	if (IsntNull(string))
		*string = NULL;

	PGPValidatePref(prefRef);
	PGPValidatePtr(prefStruct);
	PGPValidatePtr(string);

	if (memberIndex >= prefStruct->numMembers)
		return kPGPError_BadParams;

	if (prefStruct->members[memberIndex].type != kPGPPrefType_String)
		return kPGPError_BadParams;

	PGPPeekPrefMemoryMgr(prefRef, &memoryMgr);

	if (IsntNull(prefStruct->members[memberIndex].data))
	{
		*string = (char *) PGPNewData(memoryMgr,
							strlen(prefStruct->members[memberIndex].data)+1,
							kPGPMemoryMgrFlags_Clear);

		if (IsNull(*string))
			return kPGPError_OutOfMemory;

		pgpCopyMemory(prefStruct->members[memberIndex].data, *string,
			strlen(prefStruct->members[memberIndex].data) + 1);
	}

	return kPGPError_NoErr;
}


PGPError PGPSetPrefStructString(PGPPrefRef prefRef,
									PGPPrefStruct *prefStruct,
									PGPUInt16 memberIndex,
									char *string)
{
	PGPMemoryMgrRef memoryMgr;
	PGPError err = kPGPError_NoErr;

	PGPValidatePref(prefRef);
	PGPValidatePtr(prefStruct);
	PGPValidatePtr(string);

	if (memberIndex >= prefStruct->numMembers)
		return kPGPError_BadParams;

	if (prefStruct->members[memberIndex].type != kPGPPrefType_String)
		return kPGPError_BadParams;

	if (IsntNull(prefStruct->members[memberIndex].data))
		PGPFreeData(prefStruct->members[memberIndex].data);

	PGPPeekPrefMemoryMgr(prefRef, &memoryMgr);

	prefStruct->members[memberIndex].data = PGPNewData(memoryMgr,
												strlen(string) + 1,
												kPGPMemoryMgrFlags_Clear);

	if (IsNull(prefStruct->members[memberIndex].data))
		return kPGPError_OutOfMemory;

	pgpCopyMemory(string, prefStruct->members[memberIndex].data,
		strlen(string) + 1);

	prefStruct->dirty = TRUE;
	return kPGPError_NoErr;
}


PGPError PGPGetPrefStructByte(PGPPrefRef prefRef,
									PGPPrefStruct *prefStruct,
									PGPUInt16 memberIndex,
									PGPSize *dataLength,
									void **data)
{
	PGPMemoryMgrRef memoryMgr;

	if (IsntNull(dataLength))
		*dataLength = 0;
	if (IsntNull(data))
		*data = NULL;

	PGPValidatePref(prefRef);
	PGPValidatePtr(prefStruct);
	PGPValidatePtr(dataLength);
	PGPValidatePtr(data);

	if (memberIndex >= prefStruct->numMembers)
		return kPGPError_BadParams;

	if (prefStruct->members[memberIndex].type != kPGPPrefType_Byte)
		return kPGPError_BadParams;

	PGPPeekPrefMemoryMgr(prefRef, &memoryMgr);

	if (IsntNull(prefStruct->members[memberIndex].data))
	{
		*data = PGPNewData(memoryMgr,
					prefStruct->members[memberIndex].size,
					kPGPMemoryMgrFlags_Clear);

		if (IsNull(*data))
			return kPGPError_OutOfMemory;

		*dataLength = prefStruct->members[memberIndex].size;

		pgpCopyMemory(prefStruct->members[memberIndex].data, *data,
			prefStruct->members[memberIndex].size);
	}

	return kPGPError_NoErr;
}


PGPError PGPSetPrefStructByte(PGPPrefRef prefRef,
									PGPPrefStruct *prefStruct,
									PGPUInt16 memberIndex,
									PGPSize dataLength,
									void *data)
{
	PGPMemoryMgrRef memoryMgr;
	PGPError err = kPGPError_NoErr;

	PGPValidatePref(prefRef);
	PGPValidatePtr(prefStruct);

	if (memberIndex >= prefStruct->numMembers)
		return kPGPError_BadParams;

	if (prefStruct->members[memberIndex].type != kPGPPrefType_Byte)
		return kPGPError_BadParams;

	if (IsntNull(prefStruct->members[memberIndex].data))
	{
		PGPFreeData(prefStruct->members[memberIndex].data);
		prefStruct->members[memberIndex].data = NULL;
	}

	if (IsNull(data))
		return kPGPError_NoErr;

	PGPPeekPrefMemoryMgr(prefRef, &memoryMgr);

	prefStruct->members[memberIndex].data = PGPNewData(memoryMgr,
												dataLength,
												kPGPMemoryMgrFlags_Clear);

	if (IsNull(prefStruct->members[memberIndex].data))
		return kPGPError_OutOfMemory;

	pgpCopyMemory(data, prefStruct->members[memberIndex].data,
		dataLength);

	prefStruct->members[memberIndex].size = dataLength;
	prefStruct->dirty = TRUE;

	return kPGPError_NoErr;
}


/*__Editor_settings____

	Local Variables:
	tab-width: 4
	End:
	vi: ts=4 sw=4
	vim: si
_____________________*/


